import sys
import json
import os
import time
import re
import requests
import threading
import asyncio
import logging
import pyotp
from decimal import Decimal
from datetime import datetime
from PyQt6.QtGui import QTextCursor
from PyQt6.QtCore import Qt, QTimer, pyqtSignal, QObject
from PyQt6.QtMultimedia import QSoundEffect
from PyQt6.QtCore import QUrl
from PyQt6.QtWidgets import (QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QTextEdit,
                             QLineEdit, QLabel, QGridLayout, QMessageBox, QFrame, QInputDialog,
                             QCheckBox, QComboBox)

from config import BotConfig
from core.trading_bot import TradingBot, TradingState
from licensing.licensing_module import get_machine_code, load_license, check_online, save_license
from .activation_window import ActivationWindow
from core.proxy_parser import get_requests_proxies

class SignalEmitter(QObject):
    log_signal = pyqtSignal(list)
    cycle_summary_signal = pyqtSignal(dict)
    key_signal = pyqtSignal(str)
    bot_finished_signal = pyqtSignal(Decimal)
    proxy_test_finished_signal = pyqtSignal()
    update_proxy_ui_signal = pyqtSignal(str, str)
    play_sound_signal = pyqtSignal(str)

class BotGUI(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("AMGH Bot 个人多开版 16.5")
        self.setFixedSize(620, 460)
        self.DEBUG_PASSWORD = "bingo528"
        self.debug_mode_authorized = False

        self.bot_thread = None
        self.bot_running = False
        self.is_paused = False
        self.bot: TradingBot | None = None
        self.start_time = None
        self.elapsed_time = 0
        self.machine_code = get_machine_code()
        self.tokens = []
        self.log_emitter = SignalEmitter()
        self.init_ui()

        self.sounds = {
            'success': QSoundEffect(),
            'error': QSoundEffect(),
            'gogo': QSoundEffect()
        }
        try:
            # 設定音效來源，使用 .wav 檔案
            self.sounds['success'].setSource(QUrl.fromLocalFile(self.get_bundled_path('success.wav')))
            self.sounds['error'].setSource(QUrl.fromLocalFile(self.get_bundled_path('error.wav')))
            self.sounds['gogo'].setSource(QUrl.fromLocalFile(self.get_bundled_path('gogo.wav')))

            self.log([("INFO", "✅ 音訊裝置初始成功，聲音功能啟用。")])  # 現在呼叫 self.log 是安全的
            self.is_sound_available = True
        except Exception as e:
            self.log([("INFO", f"⚠️ 警告：無法初始化音訊裝置，聲音功能將被禁用: {e}")])
            self.is_sound_available = False


        self.log_emitter.log_signal.connect(self.log)
        self.log_emitter.cycle_summary_signal.connect(self.update_summary_data)
        self.log_emitter.bot_finished_signal.connect(self.on_bot_finished)
        self.btn_test_proxy.clicked.connect(self.handle_proxy_test_button_click)
        self.log_emitter.proxy_test_finished_signal.connect(lambda: self.btn_test_proxy.setEnabled(True))
        self.log_emitter.update_proxy_ui_signal.connect(self.update_proxy_ui)
        self.log_emitter.play_sound_signal.connect(self.play_sound)
        self.load_and_populate_tokens()

        self.update_timer = QTimer(self)
        self.update_timer.timeout.connect(self.update_info)
        self.info_labels["累计盈亏"].setText("待任务结束后计算")

    def get_app_base_path(self):
        if getattr(sys, 'frozen', False):
            return os.path.dirname(sys.executable)
        else:
            return os.path.abspath(os.path.join(os.path.dirname(__file__), ".."))

    def get_bundled_path(self, relative_path):
        """获取打包在可执行文件内部的资源路径。"""
        if getattr(sys, 'frozen', False):
            # PyInstaller 打包后，sys._MEIPASS 指向解压文件的临时目录
            return os.path.join(sys._MEIPASS, relative_path)
        else:
            # 源码运行，直接返回相对路径
            return os.path.join(self.get_app_base_path(), relative_path)

    def play_sound(self, sound_type: str):
        """使用 QSoundEffect 播放預載入的音效。此方法在主執行緒中執行是安全的。"""
        if not self.is_sound_available:
            return

        if sound_effect := self.sounds.get(sound_type):
            if sound_effect.isLoaded():
                sound_effect.play()
            else:
                self.log([("INFO", f"提示：音效檔 {sound_type}.wav 未能成功載入或路徑錯誤。")])

    def init_ui(self):
        main_layout = QHBoxLayout(self)
        # --- 左侧：日志输出区域 ---
        self.log_output = QTextEdit()
        self.log_output.setReadOnly(True)
        self.log_output.setFrameStyle(QFrame.Shape.Box | QFrame.Shadow.Plain)
        self.log_output.document().setMaximumBlockCount(1000)
        main_layout.addWidget(self.log_output, 10)
        # --- 右侧：所有控制控件的面板 ---
        right_layout = QVBoxLayout()
        control_frame = QFrame()
        control_layout = QGridLayout(control_frame)
        control_layout.setSpacing(10)
        # -- 代币选择 (第0行) --
        control_layout.addWidget(QLabel("选择代币:"), 0, 0, Qt.AlignmentFlag.AlignRight)
        token_selection_layout = QHBoxLayout()
        token_selection_layout.setContentsMargins(0, 0, 0, 0)
        self.token_search_input = QLineEdit()
        self.token_search_input.setPlaceholderText("搜索代币")
        self.token_search_input.textChanged.connect(self.filter_token_list)
        self.token_combo = QComboBox()
        token_selection_layout.addWidget(self.token_combo, 2)
        token_selection_layout.addWidget(self.token_search_input, 1)
        control_layout.addLayout(token_selection_layout, 0, 1)
        # -- 调试模式复选框 (第0行) --
        self.debug_checkbox = QCheckBox("启用调试")
        self.debug_checkbox.clicked.connect(self.handle_debug_clicked)
        control_layout.addWidget(self.debug_checkbox, 0, 2)
        # -- 交易参数输入 (第1行) --
        h_trade_layout = QHBoxLayout()
        h_trade_layout.setContentsMargins(0, 0, 0, 0)
        self.trade_count_input = QLineEdit()
        self.trade_count_input.setPlaceholderText("次数")
        self.trade_count_input.setFixedWidth(65)
        self.trade_amount_input = QLineEdit()
        self.trade_amount_input.setPlaceholderText("金额")
        self.trade_amount_input.setFixedWidth(65)
        self.amount_float_checkbox = QCheckBox("浮动5%")
        self.amount_float_checkbox.setChecked(True)
        h_trade_layout.addWidget(QLabel("交易次数:"))
        h_trade_layout.addWidget(self.trade_count_input)
        h_trade_layout.addWidget(QLabel("交易金额:"))
        h_trade_layout.addWidget(self.trade_amount_input)
        h_trade_layout.addWidget(self.amount_float_checkbox)
        h_trade_layout.addStretch()
        control_layout.addLayout(h_trade_layout, 1, 0, 1, 3)
        # -- 轮次间隔输入 (第3行) --
        control_layout.addWidget(QLabel("轮次间隔:"), 3, 0, Qt.AlignmentFlag.AlignRight)
        self.trade_delay_input = QLineEdit("2")
        self.trade_delay_input.setFixedWidth(202)
        self.random_delay_checkbox = QCheckBox("动态随机")
        self.random_delay_checkbox.setChecked(True)
        h_delay_layout = QHBoxLayout()
        h_delay_layout.setContentsMargins(0, 0, 0, 0)
        h_delay_layout.addWidget(self.trade_delay_input)
        h_delay_layout.addWidget(self.random_delay_checkbox)
        h_delay_layout.addStretch()
        control_layout.addLayout(h_delay_layout, 3, 1, 1, 2)
        # -- 新增：二次验证密钥输入 (第4行) --
        control_layout.addWidget(QLabel("验证密钥:"), 4, 0, Qt.AlignmentFlag.AlignRight)
        h_secret_layout = QHBoxLayout()
        h_secret_layout.setContentsMargins(0,0,0,0)
        self.secret_input = QLineEdit()
        self.secret_input.setEchoMode(QLineEdit.EchoMode.Password)
        self.secret_input.setPlaceholderText("在此输入验证器密钥")
        self.btn_test_secret = QPushButton("测试")
        self.btn_save_secret = QPushButton("保存")
        self.btn_test_secret.setFixedWidth(60)
        self.btn_save_secret.setFixedWidth(60)
        h_secret_layout.addWidget(self.secret_input)
        h_secret_layout.addWidget(self.btn_test_secret)
        h_secret_layout.addWidget(self.btn_save_secret)
        control_layout.addLayout(h_secret_layout, 4, 1, 1, 2)

        # -- 代理设置 (现在是第5行) --
        control_layout.addWidget(QLabel("代理设置:"), 5, 0, Qt.AlignmentFlag.AlignRight)
        h_proxy_layout = QHBoxLayout()
        h_proxy_layout.setContentsMargins(0, 0, 0, 0)
        self.proxy_type_combo = QComboBox()
        self.proxy_type_combo.addItems(["不使用", "HTTP", "SOCKS5"])
        self.proxy_type_combo.setFixedWidth(75)
        self.proxy_string_input = QLineEdit()
        self.proxy_string_input.setPlaceholderText("IP:端口:用户名:密码")
        self.btn_test_proxy = QPushButton("检测")
        self.btn_test_proxy.setFixedWidth(35)
        h_proxy_layout.addWidget(self.proxy_type_combo)
        h_proxy_layout.addWidget(self.proxy_string_input)
        h_proxy_layout.addWidget(self.btn_test_proxy)
        control_layout.addLayout(h_proxy_layout, 5, 1, 1, 2)

        # -- 功能按钮 (现在是第6行) --
        h_buttons = QHBoxLayout()
        self.btn_parse_headers = QPushButton("解析请求头")
        self.btn_start = QPushButton("启动交易")
        self.btn_pause = QPushButton("暂停")
        self.btn_stop = QPushButton("停止交易")
        self.btn_pause.setEnabled(False)
        self.btn_stop.setEnabled(False)
        self.btn_parse_headers.setFixedHeight(40)
        self.btn_start.setFixedHeight(40)
        self.btn_pause.setFixedHeight(40)
        self.btn_stop.setFixedHeight(40)
        h_buttons.addWidget(self.btn_parse_headers)
        h_buttons.addWidget(self.btn_start)
        h_buttons.addWidget(self.btn_pause)
        h_buttons.addWidget(self.btn_stop)
        control_layout.addLayout(h_buttons, 6, 0, 1, 3)

        # -- 按钮信号连接 --
        self.btn_parse_headers.clicked.connect(self.parse_headers_dialog)
        self.btn_save_secret.clicked.connect(self.save_totp_secret)  # 连接“保存”按钮
        self.btn_test_secret.clicked.connect(self.test_totp_secret)  # 连接“测试”按钮
        self.btn_start.clicked.connect(self.handle_start)
        self.btn_pause.clicked.connect(self.handle_pause_resume)
        self.btn_stop.clicked.connect(self.handle_stop)

        control_frame.setLayout(control_layout)
        right_layout.addWidget(control_frame)
        info_frame = QFrame()
        info_frame.setFrameStyle(QFrame.Shape.Box | QFrame.Shadow.Plain)
        info_layout = QGridLayout()
        info_layout.setVerticalSpacing(4)
        self.info_labels = {}
        original_labels = ["交易币种", "启动时间", "完成次数", "累计金额", "ALPHA四", "累计盈亏", "平均耗时",
                           "累计时长"]
        for i, label_text in enumerate(original_labels):
            info_layout.addWidget(QLabel(label_text + ":"), i, 0)
            val_label = QLabel("-")
            info_layout.addWidget(val_label, i, 1)
            self.info_labels[label_text] = val_label
        row_index = len(original_labels)
        separator = QFrame()
        separator.setFrameShape(QFrame.Shape.HLine)
        separator.setFrameShadow(QFrame.Shadow.Sunken)
        info_layout.addWidget(separator, row_index, 0, 1, 2)
        row_index += 1
        info_layout.addWidget(QLabel("用户:"), row_index, 0)
        user_val_label = QLabel("-")
        info_layout.addWidget(user_val_label, row_index, 1)
        self.info_labels["用户"] = user_val_label
        row_index += 1
        self.expiry_key_label = QLabel("到期时间:")
        self.expiry_val_label = QLabel("-")
        info_layout.addWidget(self.expiry_key_label, row_index, 0)
        info_layout.addWidget(self.expiry_val_label, row_index, 1)
        self.expiry_key_label.hide()
        self.expiry_val_label.hide()
        info_frame.setLayout(info_layout)
        right_layout.addWidget(info_frame)
        right_layout.addStretch()
        main_layout.addLayout(right_layout, 1)
        self.setLayout(main_layout)

    def log(self, messages: list):
        scroll_bar = self.log_output.verticalScrollBar()
        at_bottom = scroll_bar.value() == scroll_bar.maximum()
        cursor = self.log_output.textCursor()
        cursor.movePosition(QTextCursor.MoveOperation.End)
        for level, text in messages:
            if not self.debug_mode_authorized and level != 'SUMMARY':
                continue
            cursor.insertText(text + '\n')
        if at_bottom:
            scroll_bar.setValue(scroll_bar.maximum())

    # --- 【修改】核心启动流程 ---
    def check_activation_and_run(self):
        if not self.machine_code:
            QMessageBox.critical(self, "严重错误", "无法获取机器码，程序将退出。")
            sys.exit(1)

        # 假设这是多开版，我们从专属文件夹加载
        stored_code = load_license(app_name="alphabot_personal_multi")

        if stored_code:
            self.log([("SUMMARY", "▶️ 发现本地许可，正在联网验证...")])

            # 【关键修改】现在 check_online 返回一个元组 (success, data_or_message)
            success, data_or_message = check_online(self.machine_code, stored_code)

            # 我们需要判断元组的第一个元素 (success)
            if success:
                license_info = data_or_message  # 成功时，第二个元素是许可证数据
                self.log([("SUMMARY", "🎉 许可验证成功。")])
                self.update_license_display(license_info)
                self.show()
                threading.Thread(target=self.load_and_validate_config_sequence, daemon=True).start()
                self.update_timer.start(1000)
            else:
                # 失败时，第二个元素是具体的错误消息
                error_message = data_or_message
                QMessageBox.warning(self, "验证失败", f"{error_message}\n\n请重新激活。")
                self.show_activation_window()
        else:
            self.log([("SUMMARY", "▶️ 未发现本地许可文件，请激活软件。")])
            self.show_activation_window()

    # --- 【新增】新的、带顺序的配置加载和验证流程 ---
    def load_and_validate_config_sequence(self):
        """
        新的启动流程，按顺序执行：
        1. 加载配置文件。
        2. 加载并验证代理。
        3. 如果代理OK（或无代理），则加载并验证请求头。
        """
        path = self._get_config_path()
        if not os.path.exists(path):
            self.log_emitter.log_signal.emit([("SUMMARY", "▶️ 未找到本地配置文件。")])
            return

        try:
            with open(path, 'r', encoding='utf-8') as f:
                config_data = json.load(f)
        except Exception as e:
            self.log_emitter.log_signal.emit([("SUMMARY", f"🚨 加载本地配置失败: {e}")])
            return

        # --- 步骤1: 加载代理配置 ---
        proxy_type = config_data.get("proxy_type", "none")
        proxy_string = config_data.get("proxy_string", "")

        # 使用信号安全地更新UI
        self.log_emitter.update_proxy_ui_signal.emit(proxy_type, proxy_string)

        # --- 步骤2: 验证代理 ---
        proxy_map = {"不使用": "none", "HTTP": "http", "SOCKS5": "socks5"}
        proxy_type_text = {v: k for k, v in proxy_map.items()}.get(proxy_type)

        is_proxy_ok = False
        if proxy_type != "none" and proxy_string:
            self.log_emitter.log_signal.emit([("SUMMARY", f"▶️ 正在加载并检测 {proxy_type_text} 代理...")])
            success, message = self._test_proxy_logic(proxy_string, proxy_type_text)
            self.log_emitter.log_signal.emit([("SUMMARY", message)])
            if success:
                is_proxy_ok = True
        else:
            # 如果没有配置代理，也视为“代理OK”
            is_proxy_ok = True

        # --- 步骤3: 如果代理OK，则加载并验证请求头 ---
        if is_proxy_ok:
            if not is_proxy_ok or (proxy_type == "none" or not proxy_string):
                self.log_emitter.log_signal.emit([("SUMMARY", "▶️ 未配置代理，直接加载请求头...")])
            else:
                self.log_emitter.log_signal.emit([("SUMMARY", "▶️ 代理验证成功，开始加载请求头...")])

            BotConfig.REQUEST_HEADERS = config_data.get("request_headers", {})
            self._validate_headers_task()  # 执行请求头验证
        else:
            self.log_emitter.log_signal.emit([("SUMMARY", "🚨 代理验证失败，已停止加载请求头。")])

    # --- 【新增】用于从后台线程安全更新UI代理设置的槽函数 ---
    def update_proxy_ui(self, proxy_type: str, proxy_string: str):
        """安全地更新代理相关的UI组件"""
        self.proxy_string_input.setText(proxy_string)
        proxy_map_rev = {"none": "不使用", "http": "HTTP", "socks5": "SOCKS5"}
        self.proxy_type_combo.setCurrentText(proxy_map_rev.get(proxy_type, "不使用"))

        # 更新 BotConfig 以便后续的代理测试逻辑能获取到值
        BotConfig.PROXY_TYPE = proxy_type
        BotConfig.PROXY_STRING = proxy_string

    # --- 【修改】将文件路径和保存逻辑统一 ---
    def _get_config_path(self):
        """获取config.json的路径。"""
        base_path = self.get_app_base_path()
        return os.path.join(base_path, "config.json")

    def save_local_config(self):
        """保存所有配置（请求头和代理）到文件"""
        path = self._get_config_path()
        try:
            # 从UI获取最新的代理设置并更新到BotConfig
            proxy_map = {"不使用": "none", "HTTP": "http", "SOCKS5": "socks5"}
            BotConfig.PROXY_TYPE = proxy_map[self.proxy_type_combo.currentText()]
            BotConfig.PROXY_STRING = self.proxy_string_input.text().strip()

            config_to_save = {
                "request_headers": BotConfig.REQUEST_HEADERS,
                "listen_key": BotConfig.LISTEN_KEY,
                "proxy_type": BotConfig.PROXY_TYPE,
                "proxy_string": BotConfig.PROXY_STRING
            }
            with open(path, 'w', encoding='utf-8') as f:
                json.dump(config_to_save, f, ensure_ascii=False, indent=2)
            self.log([("SUMMARY", "✅ 当前配置已自动保存。")])
        except Exception as e:
            self.log([("SUMMARY", f"🚨 保存配置失败: {e}")])

    # --- 【修改】将原有的 validate_headers 重命名为内部任务 ---
    def _validate_headers_task(self):
        """验证请求头（获取Listen Key）的核心逻辑"""
        try:
            headers = BotConfig.REQUEST_HEADERS
            if not headers:
                self.log_emitter.log_signal.emit([("SUMMARY", "🚨 本地请求头为空，请重新解析。")])
                return

            proxies = get_requests_proxies()
            url_auth = "https://www.binance.com/bapi/defi/v1/private/alpha-trade/get-listen-key"
            resp_auth = requests.post(url_auth, headers=headers, json={}, timeout=10, verify=False, proxies=proxies)
            resp_auth.raise_for_status()
            data_auth = resp_auth.json()
            listen_key = data_auth.get("data", "")
            if listen_key:
                BotConfig.LISTEN_KEY = listen_key
                self.log_emitter.key_signal.emit(listen_key)
                self.log_emitter.log_signal.emit([("SUMMARY", "✅ 请求头验证完成，KEY 有效。")])
            else:
                raise ValueError("获取 KEY 失败。")
        except requests.exceptions.HTTPError as http_err:
            if http_err.response.status_code == 401:
                self.log_emitter.log_signal.emit([("SUMMARY", "🚨 请求头授权已过期，请在网站扫码登录后抓取。")])
            else:
                self.log_emitter.log_signal.emit([("SUMMARY", f"🚨 请求头验证失败，请检查网络情况及更换节点尝试。")])
            BotConfig.REQUEST_HEADERS, BotConfig.LISTEN_KEY = {}, ""
        except Exception as e:
            logging.error(f"验证本地请求头失败: {e}", exc_info=True)
            self.log_emitter.log_signal.emit([("SUMMARY", f"🚨 请求头验证失败，请检查网络情况及更换节点尝试。")])
            BotConfig.REQUEST_HEADERS, BotConfig.LISTEN_KEY = {}, ""

    # --- 【修改】将代理检测的核心逻辑提取出来 ---
    def _test_proxy_logic(self, proxy_string: str, proxy_type_text: str) -> tuple[bool, str]:
        """
        执行代理检测的核心网络请求逻辑。
        :return: 一个元组 (is_success, message)
        """
        test_url = 'https://api.ipify.org?format=json'
        try:
            parts = proxy_string.split(':')
            if len(parts) not in [2, 4]:
                raise ValueError("格式应为 IP:Port 或 IP:Port:User:Pass")
            ip, port = parts[0], parts[1]
            protocol = {"HTTP": "http", "SOCKS5": "socks5h"}.get(proxy_type_text)
            if len(parts) == 4:
                proxy_url = f"{protocol}://{parts[2]}:{parts[3]}@{ip}:{port}"
            else:
                proxy_url = f"{protocol}://{ip}:{port}"
            proxies = {"http": proxy_url, "https": proxy_url}
            response = requests.get(test_url, proxies=proxies, timeout=10, verify=False)
            response.raise_for_status()
            detected_ip = response.json().get("ip")
            return True, f"✅ 代理可用。出口 IP: {detected_ip}"
        except Exception as e:
            return False, f"🚨 代理连接失败: {str(e)}"

    # --- 【修改】处理按钮点击的函数，现在它调用核心逻辑 ---
    def handle_proxy_test_button_click(self):
        """处理“检测”按钮的点击事件"""
        proxy_string = self.proxy_string_input.text().strip()
        proxy_type_text = self.proxy_type_combo.currentText()
        if proxy_type_text == "不使用" or not proxy_string:
            self.log([("SUMMARY", "ℹ️ 未设置代理，无需检测。")])
            return
        self.log([("SUMMARY", f"▶️ 正在检测 {proxy_type_text} 代理...")])
        self.btn_test_proxy.setEnabled(False)
        threading.Thread(target=self.run_proxy_check_for_button, args=(proxy_string, proxy_type_text),
                         daemon=True).start()

    def run_proxy_check_for_button(self, proxy_string: str, proxy_type_text: str):
        """专门为按钮点击事件启动的线程任务"""
        # 更新BotConfig，因为手动检测时UI可能是最新的
        proxy_map = {"不使用": "none", "HTTP": "http", "SOCKS5": "socks5"}
        BotConfig.PROXY_TYPE = proxy_map[proxy_type_text]
        BotConfig.PROXY_STRING = proxy_string

        success, message = self._test_proxy_logic(proxy_string, proxy_type_text)
        self.log_emitter.log_signal.emit([("SUMMARY", message)])
        self.log_emitter.proxy_test_finished_signal.emit()

    # --- 其他函数保持不变或做微小调整 ---
    def parse_headers_task(self, headers_text):
        try:
            # 在解析前，先从UI更新BotConfig中的代理设置
            proxy_map = {"不使用": "none", "HTTP": "http", "SOCKS5": "socks5"}
            BotConfig.PROXY_TYPE = proxy_map[self.proxy_type_combo.currentText()]
            BotConfig.PROXY_STRING = self.proxy_string_input.text().strip()
            proxies = get_requests_proxies()

            self.log_emitter.log_signal.emit([("SUMMARY", "▶️ 正在后台解析请求头...")])
            headers = self._internal_parse_headers(headers_text)
            if not headers: raise ValueError("无法解析出有效的请求头。")
            url_auth = "https://www.binance.com/bapi/defi/v1/private/alpha-trade/get-listen-key"
            resp_auth = requests.post(url_auth, headers=headers, json={}, timeout=10, verify=False, proxies=proxies)
            resp_auth.raise_for_status()
            data_auth = resp_auth.json()
            BotConfig.REQUEST_HEADERS = headers
            BotConfig.LISTEN_KEY = data_auth.get("data", "")
            if BotConfig.LISTEN_KEY:
                self.log_emitter.key_signal.emit(BotConfig.LISTEN_KEY)
                self.log_emitter.log_signal.emit([("SUMMARY", "✅ KEY 获取完成")])
                self.save_local_config()  # 【修改】调用新的保存函数
            else:
                raise ValueError("获取 KEY 失败。")
        except requests.exceptions.HTTPError as http_err:
            BotConfig.REQUEST_HEADERS, BotConfig.LISTEN_KEY = {}, ""
            if http_err.response.status_code == 401:
                self.log_emitter.log_signal.emit([("SUMMARY", "🚨 请求头授权已过期，请在网站扫码登录后抓取。")])
            else:
                self.log_emitter.log_signal.emit([("SUMMARY", f"🚨 解析请求头失败:请检查网络情况及更换节点尝试")])
        except Exception as e:
            BotConfig.REQUEST_HEADERS, BotConfig.LISTEN_KEY = {}, ""
            self.log_emitter.log_signal.emit([("SUMMARY", f"🚨 解析请求头失败:请检查网络情况及更换节点尝试")])

    def handle_start(self):
        if self.bot_running:
            QMessageBox.warning(self, "提示", "交易已经在运行中。");
            return

        if self.token_combo.currentData() is None:
            QMessageBox.warning(self, "提示", "请选择代币。")
            return

        self.info_labels["累计盈亏"].setStyleSheet("color: black;")
        self.info_labels["累计盈亏"].setText("运行中...")
        try:
            BotConfig.TOTAL_CYCLES = int(self.trade_count_input.text().strip())
            BotConfig.TRADE_AMOUNT_USDT = Decimal(self.trade_amount_input.text().strip())
            BotConfig.FIXED_DELAY_SECONDS = float(self.trade_delay_input.text().strip())
            BotConfig.RANDOM_DELAY_ENABLED = self.random_delay_checkbox.isChecked()
            BotConfig.AMOUNT_FLOAT_ENABLED = self.amount_float_checkbox.isChecked()
        except Exception:
            QMessageBox.warning(self, "提示", "交易参数格式不正确，必须是数字。");
            return
        # 第一步：检查请求头和API Key相关的核心配置
        if not all([BotConfig.REQUEST_HEADERS, BotConfig.LISTEN_KEY]):
            QMessageBox.warning(self, "提示", "缺少必要配置，请先解析请求头。")
            return

        # 第二步：独立检查是否已选择交易代币
        if not BotConfig.SYMBOL:
            QMessageBox.warning(self, "提示", "请选择要交易的代币。\n如搜索的代币名称少于3个字母，请在列表中手动选择。")
            return
        BotConfig.PUBLIC_STREAM_NAME = BotConfig.BASE_ASSET.lower() + "usdt@aggTrade"
        BotConfig.CANCEL_ID = BotConfig.BASE_ASSET + "USDT"
        # --- 新增：交易任务确认弹窗 ---
        token_name = BotConfig.SYMBOL
        cycles = BotConfig.TOTAL_CYCLES
        amount_per_trade = BotConfig.TRADE_AMOUNT_USDT
        total_volume = cycles * amount_per_trade
        four_x_volume = total_volume * 4
        # 建议金额使用和机器人启动时检查逻辑一致的 1.5 倍
        suggested_balance = amount_per_trade * Decimal("1.5")

        # 使用 f-string 格式化将要显示的信息
        confirmation_message = f"""
           请确认您的交易任务信息：

        🔹 **交易代币**:  {token_name}
        🔄 **交易次数**:  {cycles} 次
        💵 **单笔金额**:  {amount_per_trade:.2f} USDT

        --------------------------------

        📈 **预计总量**:  {total_volume:.2f} USDT
        🚀 **预计四倍**:  {four_x_volume:.2f} USDT
        💰 **建议资金**:  {suggested_balance:.2f} USDT

        点击“确认”后将立即开始执行。
        """

        msg_box = QMessageBox(self)
        msg_box.setWindowTitle("确认交易任务")
        msg_box.setIcon(QMessageBox.Icon.Information)
        # 支持富文本，让排版更好看
        msg_box.setTextFormat(Qt.TextFormat.MarkdownText)
        msg_box.setText(confirmation_message)

        # 自定义按钮文字
        confirm_button = msg_box.addButton("确认 (继续执行)", QMessageBox.ButtonRole.YesRole)
        cancel_button = msg_box.addButton("取消 (调整设置)", QMessageBox.ButtonRole.NoRole)

        # 显示弹窗
        msg_box.exec()

        # 检查用户点击了哪个按钮
        if msg_box.clickedButton() == cancel_button:
            return  # 如果用户点击取消，则中止启动流程
        # --- 新增结束 ---

        # 【新增】播放开始声音。因为这个函数在主线程中，所以可以直接调用。
        self.play_sound('gogo')

        self.log([("SUMMARY", "▶️ 正在初始化并启动交易核心...")])
        self.bot = TradingBot(gui_emitter=self.log_emitter)
        self.bot_running = True
        self.is_paused = False
        self.btn_start.setEnabled(False)
        self.btn_stop.setEnabled(True)
        self.btn_pause.setEnabled(True)
        self.btn_pause.setText("暂停")
        self.start_time = datetime.now()
        self.elapsed_time = 0
        self.bot_thread = threading.Thread(target=self.run_bot_async, daemon=True)
        self.bot_thread.start()

    # ... 以下是其他未发生重大逻辑变化的辅助函数 ...
    def show_activation_window(self):
        self.hide()
        activation_window = ActivationWindow(self)
        if activation_window.exec():
            self.log([("SUMMARY", "🎉 激活成功，正在重新加载和验证...")])
            self.check_activation_and_run()
        else:
            sys.exit(0)

    def load_tokens_from_web(self):
        url = "https://download.bingoweb3.space/tokens.json"
        local_path = os.path.join(self.get_app_base_path(), "tokens_cache.json")
        try:
            proxies = get_requests_proxies()
            resp = requests.get(url, timeout=5, verify=False, proxies=proxies)
            resp.raise_for_status()
            tokens = resp.json()
            with open(local_path, 'w', encoding='utf-8') as f:
                json.dump(tokens, f, ensure_ascii=False, indent=2)
            return tokens
        except Exception as e:
            self.log([("SUMMARY", f"⚠️ 下载代币列表失败: {e}，尝试加载本地缓存")])
            if os.path.exists(local_path):
                with open(local_path, 'r', encoding='utf-8') as f:
                    return json.load(f)
            return []

    def update_license_display(self, license_info):
        if not license_info: return
        username = license_info.get('username', '未知')
        l_type = license_info.get('license_type', 'N/A')
        expires_at = license_info.get('expires_at')
        from datetime import timezone, timedelta
        if l_type == 'monthly':
            type_text = "月卡"
        elif l_type == 'trial':
            type_text = "试用卡"
        else:
            type_text = "终身卡"
        user_display_text = f"{username} ({type_text})"
        self.info_labels["用户"].setText(user_display_text)
        if l_type in ['monthly', 'trial'] and expires_at:
            dt_obj_utc = datetime.fromisoformat(expires_at)
            if dt_obj_utc.tzinfo is None: dt_obj = dt_obj_utc.replace(tzinfo=timezone.utc)
            cst_timezone = timezone(timedelta(hours=8))
            dt_obj_cst = dt_obj.astimezone(cst_timezone)
            if l_type == 'trial':
                display_format = "%Y-%m-%d %H:%M (UTC+8)"
            else:
                display_format = "%Y-%m-%d"
            self.expiry_val_label.setText(dt_obj_cst.strftime(display_format))
            self.expiry_key_label.show()
            self.expiry_val_label.show()
        else:
            self.expiry_key_label.hide()
            self.expiry_val_label.hide()

    def handle_token_selected(self, index):
        if index < 0: return
        selected_token = self.token_combo.itemData(index)
        if not selected_token:
            self.info_labels["交易币种"].setText("-")
            BotConfig.SYMBOL = ""
            return
        BotConfig.SYMBOL = selected_token['symbol']
        BotConfig.BASE_ASSET = selected_token['base_asset']
        BotConfig.CANCEL_ID = selected_token['base_asset'].upper() + "USDT"
        self.info_labels["交易币种"].setText(BotConfig.SYMBOL)
        self.log_emitter.log_signal.emit([("SUMMARY", f"🔹 选择代币: {BotConfig.SYMBOL}")])

    def update_delay_input_state(self):
        self.trade_delay_input.setEnabled(not self.random_delay_checkbox.isChecked())

    def format_duration(self, total_seconds):
        seconds_int = int(total_seconds)
        hours, minutes, seconds = seconds_int // 3600, (seconds_int % 3600) // 60, seconds_int % 60
        parts = []
        if hours > 0: parts.append(f"{hours} 小时")
        if minutes > 0: parts.append(f"{minutes} 分钟")
        if seconds > 0 or not parts: parts.append(f"{seconds} 秒")
        return " ".join(parts) if parts else "0 秒"

    def handle_debug_clicked(self, is_checked):
        if is_checked:
            password, ok = QInputDialog.getText(self, "需要授权", "请输入调试模式密码:", QLineEdit.EchoMode.Password)
            if ok and password == self.DEBUG_PASSWORD:
                self.debug_mode_authorized = True; self.log([("INFO", "调试模式已开启。")])
            else:
                self.debug_mode_authorized = False; self.debug_checkbox.setChecked(False);
            if ok and password != self.DEBUG_PASSWORD: QMessageBox.warning(self, "错误", "密码不正确！")
        else:
            self.debug_mode_authorized = False; self.log([("INFO", "调试模式已关闭。")])

    def handle_pause_resume(self):
        if not self.bot_running or not self.bot: return
        self.is_paused = not self.is_paused
        if self.is_paused:
            self.bot.pause(); self.btn_pause.setText("继续")
        else:
            self.bot.resume(); self.btn_pause.setText("暂停")

    def run_bot_async(self):
        final_pnl = Decimal("0")
        try:
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)
            loop.run_until_complete(self.bot.start())

            # 仅在机器人正常跑完所有轮次后才查询余额
            if self.bot and self.bot.current_state == TradingState.STOPPED_SUCCESS:
                self.log([("SUMMARY", "等待10秒查询最终余额。")])
                time.sleep(10)
                self.bot.final_usdt_balance = self.bot.api_client.get_funding_wallet_usdt_balance()
                final_pnl = self.bot.final_usdt_balance - self.bot.initial_usdt_balance

            loop.run_until_complete(self.bot.shutdown())
        except Exception as e:
            self.log_emitter.log_signal.emit([("INFO", f"⚠️ 交易线程异常: {e}")])
        finally:
            # --- 声音播放逻辑，现在通过信号实现 ---
            if self.bot and self.bot.current_state == TradingState.STOPPED_SUCCESS:
                self.log_emitter.play_sound_signal.emit('success')
            else:
                self.log_emitter.play_sound_signal.emit('error')
            # -------------------------------------
            self.log_emitter.bot_finished_signal.emit(final_pnl)

    def on_bot_finished(self, final_pnl: Decimal):
        self.bot_running = False
        self.is_paused = False
        self.btn_start.setEnabled(True)
        self.btn_stop.setEnabled(False)
        self.btn_pause.setEnabled(False)
        self.btn_pause.setText("暂停")
        self.btn_stop.setText("停止交易")
        color = "green" if final_pnl >= 0 else "red"
        pnl_text = f"{final_pnl:+.4f}"
        self.info_labels["累计盈亏"].setStyleSheet(f"color: {color};")
        self.info_labels["累计盈亏"].setText(pnl_text)
        self.log([("SUMMARY", "✅ 交易任务已完全结束。")])

    def handle_stop(self):
        if not self.bot_running or not self.bot: return
        self.bot.stop()
        self.btn_stop.setText("正在停止...")
        self.btn_stop.setEnabled(False)
        self.btn_pause.setEnabled(False)

    def update_summary_data(self, data: dict):
        completed = data.get("completed_cycles", 0)
        total = data.get("total_cycles", 0)
        total_volume = data.get("total_trade_volume", Decimal("0"))
        self.info_labels["完成次数"].setText(f"{completed}/{total}")
        self.info_labels["累计金额"].setText(f"{total_volume:.2f}")
        self.info_labels["ALPHA四"].setText(f"{(total_volume * 4):.2f} （详见APP数据）")

    def update_info(self):
        if self.bot_running and self.start_time: self.elapsed_time = (datetime.now() - self.start_time).total_seconds()
        completed_cycles_str = self.info_labels["完成次数"].text().split('/')[0]
        completed_cycles = int(completed_cycles_str) if completed_cycles_str.isdigit() else 0
        avg_time = self.elapsed_time / completed_cycles if completed_cycles > 0 else 0
        self.info_labels["交易币种"].setText(BotConfig.SYMBOL or "-")
        self.info_labels["启动时间"].setText(self.start_time.strftime("%Y-%m-%d %H:%M:%S") if self.start_time else "-")
        self.info_labels["平均耗时"].setText(f"{avg_time:.2f} 秒")
        self.info_labels["累计时长"].setText(self.format_duration(self.elapsed_time))

    def _internal_parse_headers(self, headers_text: str) -> dict:
        headers, i = {}, 0
        cleaned_text = headers_text.strip()
        if cleaned_text.startswith('(') and ')' in cleaned_text:
            start_index, end_index = cleaned_text.find('\n'), cleaned_text.rfind(')')
            if -1 < start_index < end_index: cleaned_text = cleaned_text[start_index:end_index].strip()
        lines = [line.strip() for line in cleaned_text.splitlines() if line.strip()]
        while i < len(lines):
            line = lines[i]
            if line.startswith(':'): i += 1; continue
            parts = line.split(':', 1)
            if len(parts) == 2:
                key, value = parts[0].strip(), parts[1].strip()
                if not value and (i + 1) < len(lines) and not lines[i + 1].startswith(':'):
                    headers[key] = lines[i + 1].strip(); i += 2
                else:
                    headers[key] = value; i += 1
            elif (i + 1) < len(lines) and not lines[i + 1].startswith(':'):
                headers[parts[0].strip()] = lines[i + 1].strip(); i += 2
            else:
                i += 1
        return headers

    def parse_headers_dialog(self):
        text, ok = QInputDialog.getMultiLineText(self, "粘贴请求头", "请粘贴完整请求头：")
        if ok and text.strip(): threading.Thread(target=self.parse_headers_task, args=(text,), daemon=True).start()

    def load_and_populate_tokens(self):
        self.tokens = self.load_tokens_from_web()
        self.token_combo.clear()
        if not self.tokens: self.token_combo.addItem("加载失败", None); return
        self.token_combo.addItem("--选择代币--", None)
        for token in self.tokens:
            self.token_combo.addItem(f"{token['name']}", token)
        try:
            self.token_combo.currentIndexChanged.disconnect(self.handle_token_selected)
        except TypeError:
            pass
        self.token_combo.currentIndexChanged.connect(self.handle_token_selected)

    def filter_token_list(self):
        search_text = self.token_search_input.text().lower()
        try:
            self.token_combo.currentIndexChanged.disconnect(self.handle_token_selected)
        except TypeError:
            pass
        self.token_combo.clear()
        if not search_text: self.load_and_populate_tokens(); return
        filtered_tokens = [t for t in self.tokens if t['symbol'].lower().startswith(search_text)]
        if not filtered_tokens:
            self.token_combo.addItem("未找到", None)
        else:
            for token in filtered_tokens: self.token_combo.addItem(f"{token['name']}", token)
        self.token_combo.currentIndexChanged.connect(self.handle_token_selected)
        if len(filtered_tokens) == 1 and filtered_tokens[0]['symbol'].lower() == search_text:
            self.token_combo.setCurrentIndex(0)
            self.handle_token_selected(0)

    def save_totp_secret(self):
        """
        保存用户输入的TOTP密钥到BotConfig中。
        """
        secret_text = self.secret_input.text().strip().upper()
        if not secret_text:
            BotConfig.TOTP_SECRET = ""
            self.log([("SUMMARY", "验证码密钥已清空。")])
            QMessageBox.information(self, "提示", "验证码密钥已清空。")
            return

        if re.match(r'^[A-Z2-7]{16,}$', secret_text):
            BotConfig.TOTP_SECRET = secret_text
            self.log([("SUMMARY", f"✅ 验证码密钥已在程序中设置。")])
            QMessageBox.information(self, "成功", "验证码密钥已设置成功！\n注意：关闭程序后需要重新设置。")
        else:
            QMessageBox.warning(self, "格式错误", "密钥格式似乎不正确。\n请确保输入了正确的Google Authenticator密钥。")

    def test_totp_secret(self):
        """
        测试用户输入的密钥，并在弹窗中正确显示带格式的实时倒计时。
        """
        secret_text = self.secret_input.text().strip().upper()
        if not re.match(r'^[A-Z2-7]{16,}$', secret_text):
            QMessageBox.warning(self, "格式错误", "请输入有效的密钥后再进行测试。")
            return

        try:
            totp = pyotp.TOTP(secret_text)

            msg_box = QMessageBox(self)
            msg_box.setWindowTitle("验证码测试")
            msg_box.setIcon(QMessageBox.Icon.Information)

            # 关键修复：告诉弹窗要解析HTML格式
            msg_box.setTextFormat(Qt.TextFormat.RichText)

            msg_box.setStandardButtons(QMessageBox.StandardButton.Ok)

            expiry_time = [0]

            def update_display():
                current_timestamp = int(time.time())

                if current_timestamp >= expiry_time[0]:
                    new_code = totp.now()
                    expiry_time[0] = (current_timestamp // 30 + 1) * 30
                else:
                    new_code = totp.now()

                remaining_seconds = expiry_time[0] - current_timestamp
                if remaining_seconds < 0: remaining_seconds = 0

                msg_box.setText(
                    f"<h2>当前生成的验证码是: <font color='blue'>{new_code}</font></h2>"
                    f"请立即与您身份验证器中显示的6位数字进行核对。<br><br>"
                    f"如果一致，说明密钥有效！<br><br>"
                    f"<b><font color='red'>此验证码将在 {remaining_seconds} 秒后更新</font></b>"
                )

            timer = QTimer(self)
            timer.timeout.connect(update_display)
            timer.start(1000)

            update_display()
            msg_box.exec()
            timer.stop()

        except Exception as e:
            QMessageBox.critical(self, "生成失败", f"无法根据您输入的密钥生成验证码，请检查密钥是否正确。\n错误: {e}")