# main.py
import sys
import logging
import struct
from PyQt6.QtWidgets import QApplication, QMessageBox
from PyQt6.QtCore import QSharedMemory

# 从我们重构的模块中导入主窗口
from gui.main_window import BotGUI

def setup_logging():
    """配置全局日志记录器。"""
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(levelname)s - %(message)s',
        handlers=[
            logging.StreamHandler()  # 输出到控制台
        ]
    )

class InstanceLimiter:
    """基于 QSharedMemory 的实例数量限制器"""
    def __init__(self, key: str, max_instances: int):
        self.max_instances = max_instances
        self.shared = QSharedMemory(key)
        if not self.shared.attach():
            # 第一次运行时创建 4 字节存放计数器
            self.shared.create(4)

    def _read_count(self) -> int:
        self.shared.lock()
        try:
            if self.shared.size() < 4:
                return 0
            data = bytes(self.shared.data()[:4])
            return struct.unpack("I", data)[0]
        finally:
            self.shared.unlock()

    def _write_count(self, val: int):
        self.shared.lock()
        try:
            self.shared.data()[:4] = struct.pack("I", val)
        finally:
            self.shared.unlock()

    def try_acquire(self) -> bool:
        """尝试占用一个实例名额，超过限制返回 False"""
        count = self._read_count()
        if count >= self.max_instances:
            return False
        self._write_count(count + 1)
        return True

    def release(self):
        """释放一个实例名额"""
        count = self._read_count()
        self._write_count(max(0, count - 1))


if __name__ == "__main__":
    # 1. 配置日志
    setup_logging()
    # 2. 创建Qt应用实例
    app = QApplication(sys.argv)

    # 3. 设置最多允许 4 个实例运行
    MAX_INSTANCES = 4
    limiter = InstanceLimiter("BinanceAlphaBot_shared_counter", MAX_INSTANCES)

    if not limiter.try_acquire():
        msg_box = QMessageBox()
        msg_box.setIcon(QMessageBox.Icon.Warning)
        msg_box.setWindowTitle("窗口已达上限")
        msg_box.setText(f"此版本最多只能运行 {MAX_INSTANCES} 个窗口。")
        msg_box.exec()
        sys.exit(0)

    # 4. 创建并显示主窗口
    gui = BotGUI()
    gui.check_activation_and_run()

    # 退出时释放计数
    app.aboutToQuit.connect(limiter.release)

    # 5. 启动Qt事件循环
    sys.exit(app.exec())
