# gui/activation_window.py

import sys
import threading
from PyQt6.QtWidgets import QDialog, QVBoxLayout, QFormLayout, QLineEdit, QLabel, QPushButton, QMessageBox
from PyQt6.QtCore import Qt, pyqtSignal

# 注意引用路径的变化
from licensing.licensing_module import activate_online, save_license

class ActivationWindow(QDialog):
    activation_completed = pyqtSignal(bool)

    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("软件激活")
        self.setFixedSize(400, 250)
        main_layout = QVBoxLayout(self)
        form_layout = QFormLayout()
        self.user_entry = QLineEdit(self)
        self.user_entry.setPlaceholderText("仅在首次激活时设置，后续无法更改")
        form_layout.addRow("请输入您的用户名:", self.user_entry)
        self.code_entry = QLineEdit(self)
        self.code_entry.setPlaceholderText("请输入您的激活码")
        form_layout.addRow("请输入您的激活码:", self.code_entry)
        main_layout.addLayout(form_layout)
        info_label = QLabel("请勿重复激活\n如已激活过出现该提示，请检查您的网络。", self)
        info_label.setStyleSheet("color: grey;")  # 將文字設定為灰色以作區分
        info_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        main_layout.addWidget(info_label)
        self.status_label = QLabel("", self)
        self.status_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        main_layout.addWidget(self.status_label)
        activate_button = QPushButton("激活", self)
        activate_button.clicked.connect(self.activate_app)
        main_layout.addWidget(activate_button)
        self.activation_completed.connect(self.handle_activation_result)

    def activate_app(self):
        activation_code = self.code_entry.text().strip()
        username = self.user_entry.text().strip()
        if not activation_code or not username:
            QMessageBox.warning(self, "输入错误", "用户名和激活码均不能为空。")
            return
        self.status_label.setText("正在激活，请稍候...")
        # 激活逻辑需要机器码，我们从父窗口获取
        machine_code = self.parent().machine_code
        thread = threading.Thread(
            target=self.run_activation_in_thread,
            args=(machine_code, activation_code, username),
            daemon=True
        )
        thread.start()

    def run_activation_in_thread(self, machine_code, activation_code, username):
        license_data = activate_online(machine_code, activation_code, username)
        if license_data:
            save_license(activation_code)
            self.activation_completed.emit(True)
        else:
            self.activation_completed.emit(False)

    def handle_activation_result(self, success):
        if success:
            QMessageBox.information(self, "成功", "激活成功！")
            self.accept()
        else:
            self.status_label.setText("激活失败，请检查信息或联系管理员。")
            QMessageBox.critical(self, "错误", "激活码无效或用户名不符合要求，请重试。\n\n如已激活过出现该提示，请检查您的网络。")